"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.importTextRoute = importTextRoute;
var _configSchema = require("@osd/config-schema");
var _constants = require("../../common/constants");
var _util = require("../utils/util");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function importTextRoute(router, config, fileProcessors, dataSourceEnabled) {
  router.post({
    path: '/api/data_importer/_import_text',
    validate: {
      query: _configSchema.schema.object({
        fileType: _configSchema.schema.string({
          validate(value) {
            if (!config.enabledFileTypes.includes(value)) {
              return `must be an enabled file type`;
            }
          }
        }),
        indexName: _configSchema.schema.string({
          minLength: 1
        }),
        createMode: _configSchema.schema.boolean(),
        delimiter: _configSchema.schema.maybe(_configSchema.schema.string({
          validate(value) {
            if (!_constants.CSV_SUPPORTED_DELIMITERS.includes(value)) {
              return `must be a supported delimiter`;
            }
          }
        })),
        dataSource: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      body: _configSchema.schema.object({
        text: _configSchema.schema.string({
          minLength: 1,
          maxLength: config.maxTextCount
        }),
        mapping: _configSchema.schema.maybe(_configSchema.schema.string({
          minLength: 1
        }))
      })
    }
  }, async (context, request, response) => {
    const processor = fileProcessors.getFileProcessor(request.query.fileType);
    if (!processor || !processor.validateText || !processor.ingestText) {
      return response.badRequest({
        body: `${request.query.fileType} is not a registered or supported filetype`
      });
    }
    const client = await (0, _util.decideClient)(dataSourceEnabled, context, request.query.dataSource);
    if (!!!client) {
      return response.notFound({
        body: 'Data source is not enabled or does not exist'
      });
    }
    try {
      const indexExists = await client.indices.exists({
        index: request.query.indexName
      });
      if (!request.query.createMode && !indexExists.body) {
        return response.notFound({
          body: `Index ${request.query.indexName} does not exist`
        });
      }
      if (request.query.createMode && indexExists.body) {
        return response.badRequest({
          body: `Index ${request.query.indexName} already exists`
        });
      }
    } catch (e) {
      return response.internalError({
        body: `Error checking if index exists: ${e}`
      });
    }
    if (request.query.createMode) {
      const mapping = request.body.mapping;
      try {
        await client.indices.create({
          index: request.query.indexName,
          ...(mapping && {
            body: {
              mappings: JSON.parse(mapping)
            }
          })
        });
      } catch (e) {
        return response.internalError({
          body: `Error creating index: ${e}`
        });
      }
    }
    let isValid;
    try {
      isValid = await processor.validateText(request.body.text, {
        delimiter: request.query.delimiter
      });
    } catch (e) {
      return response.badRequest({
        body: `Text is not valid: ${e}`
      });
    }
    if (!isValid) {
      return response.badRequest({
        body: 'Text is not valid'
      });
    }
    try {
      const message = await processor.ingestText(request.body.text, {
        indexName: request.query.indexName,
        client,
        delimiter: request.query.delimiter,
        dataSourceId: request.query.dataSource
      });
      return response.ok({
        body: {
          message,
          success: message.failedRows.length < 1
        }
      });
    } catch (e) {
      return response.internalError({
        body: `Error ingesting text: ${e}`
      });
    }
  });
}