!===============================================================================
! Copyright 2023 Intel Corporation.
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

!   Content : Intel(R) oneAPI Math Kernel Library (oneMKL) IE SpBLAS Fortran
!             native example for MKL_SPARSE_SP2M
!
!             List of oneMKL routines used in the example:
!             - DDOT
!             - MKL_SPARSE_DESTROY
!             - MKL_SPARSE_D_CREATE_CSR
!             - MKL_SPARSE_D_EXPORT_CSR
!             - MKL_SPARSE_D_MV
!             - MKL_SPARSE_OPTIMIZE
!             - MKL_SPARSE_SET_MV_HINT
!             - MKL_SPARSE_SP2M
!
!*******************************************************************************
!
! Consider the matrices A
!
!                 |  10     11      0     0     0   |
!                 |   0      0     12    13     0   |
!   A    =        |  15      0      0     0    14   |,
!                 |   0     16     17     0     0   |
!                 |   0      0      0    18    19   |
!
! and B
!
!                 |   5      0      0     0     1   |
!                 |   0      6      0     0     0   |
!   B    =        |   0      0      7     0     0   |.
!                 |   0      0      0     8     0   |
!                 |   0      0      0     0     9   |
!
! Both matrices A and B are stored in a zero-based compressed sparse row (CSR) 
! storage scheme with three arrays (see 'Sparse Matrix Storage Schemes' in the
! Intel oneMKL Developer Reference) as follows:
!
!           values_A = ( 10  11  12  13  15  14  16  17  18  19 )
!          columns_A = (  0   1   2   3   0   4   1   2   3   4 )
!         rowIndex_A = (  0       2       4       6       8      10 )
!
!           values_B = ( 5  1  6  7  8  9  )
!          columns_B = ( 0  4  1  2  3  4  )
!         rowIndex_B = ( 0     2  3  4  5  6 )
!
!
! The examples performs the following operations:
!     1. Task 1 defines C_1 = A*B and computes two scalar products :
!
!        < C_1*x ,       y > = left,   using MKL_SPARSE_SP2M and DDOT.
!        < B*x   , (A^t)*y > = right,  using MKL_SPARSE_D_MV and DDOT.
!
!        These products should result in the same value. To display matrix C_1,
!        MKL_SPARSE_D_EXPORT_CSR is used and the result is printed.
!
!        The task uses the two-stage algorithm by calling MKL_SPARSE_SP2M
!        twice:
!        - first to allocate the row_start/row_end arrays (request =
!          SPARSE_STAGE_NNZ_COUNT)
!        - then to allocate and compute the column indices and values of non-
!          zero elements (request = SPARSE_STAGE_FINALIZE_MULT)
!
!     2. Task 2 defines C_2 = A*(B^t) and computes two scalar products :
!
!        <   C_2*x ,       y > = left,   using MKL_SPARSE_SP2M and DDOT.
!        < (B^t)*x , (A^t)*y > = right,  using MKL_SPARSE_D_MV and DDOT.
!
!        These products should result in the same value. To display matrix C_2,
!        MKL_SPARSE_D_EXPORT_CSR is used and the result is printed.
!
!        The task allocates and computes the entire output matrix in a single
!        call to MKL_SPARSE_SP2M (request = SPARSE_STAGE_FULL_MULT)
!
!*******************************************************************************
PROGRAM SPARSE_SP2M

    USE MKL_SPBLAS
    USE ISO_C_BINDING
    IMPLICIT NONE
!   *****************************************************************************
!   Sparse representation of the matrices A and B:
!   *****************************************************************************
    INTEGER, ALLOCATABLE :: csrColInd_A(:), csrRowPtr_A(:), csrColInd_B(:), csrRowPtr_B(:)
    DOUBLE PRECISION, ALLOCATABLE :: csrVal_A(:), csrVal_B(:)
    DOUBLE PRECISION, ALLOCATABLE :: x(:), y(:), rslt_mv(:), rslt_mv_trans(:)
!   CSR matrix structure
    TYPE(SPARSE_MATRIX_T) csrA, csrB, csrC_1, csrC_2
!   Variables used for exporting sparse matrix
    INTEGER        :: nrows, ncols
    INTEGER(C_INT) :: indexing, opA, opB, request
    TYPE(C_PTR)    :: rows_start_c, rows_end_c, col_indx_c, values_c
    INTEGER         , POINTER :: rows_start_f(:), rows_end_f(:), col_indx_f(:)
    DOUBLE PRECISION, POINTER :: values_f(:)
!   Other variables
    TYPE(MATRIX_DESCR) descr_type_gen
    DOUBLE PRECISION :: alpha, beta, left, right, residual_1, residual_2
    INTEGER exit_status
!   External DDOT function
    external DDOT
    DOUBLE PRECISION :: DDOT
!   *****************************************************************************
!   Declaration of local variables:
!   *****************************************************************************
    INTEGER M, NNZ_A, NNZ_B, i, ii, j, info
    M = 5
    NNZ_A = 13
    NNZ_B = 6
    descr_type_gen % TYPE = SPARSE_MATRIX_TYPE_GENERAL
    alpha = 1.0
    beta = 0.0

!   Matrix A
    allocate(csrColInd_A(NNZ_A))
    allocate(csrRowPtr_A(M+1))
    allocate(csrVal_A(NNZ_A))

    do i = 1, NNZ_A
        csrVal_A(i) = i + 9
    enddo
    do i = 1, NNZ_A
        csrColInd_A(i) = MOD(i-1, 5)
    enddo
    csrRowPtr_A(1) = 0
    do i = 2, M+1
        csrRowPtr_A(i) = csrRowPtr_A(i-1) + 2
    enddo

!   Matrix B
    allocate(csrColInd_B(NNZ_B))
    allocate(csrRowPtr_B(M+1))
    allocate(csrVal_B(NNZ_B))

    ii = 1
    csrRowPtr_B(ii) = 0
    do i = 1, M
        csrVal_B(ii) = i + 4
        csrColInd_B(ii) = i-1
        ii = ii+1
        csrRowPtr_B(i+1) = csrRowPtr_B(i) + 1
        if (i .eq. 1) then
            csrVal_B(ii) = 1
            csrColInd_B(ii) = M-1
            ii = ii+1
            csrRowPtr_B(i+1) = csrRowPtr_B(i+1) + 1
        endif
    enddo

    print*,'Input matrices for example program SPARSE_SP2M'
    print*,'---------------------------------------------------'
    print*,'Matrix A:'
    do i = 1, M
        print*,'row #',i
        do j = csrRowPtr_A(i)+1, csrRowPtr_A(i+1)
            print*,csrColInd_A(j),csrVal_A(j)
        enddo
    enddo
         
    print*,'Matrix B:'
    do i = 1, M
        print*,'row #',i
        do j = csrRowPtr_B(i)+1, csrRowPtr_B(i+1)
            print*,csrColInd_B(j),csrVal_B(j)
        enddo
    enddo

!   Vectors x and y
    allocate(x(M))
    allocate(y(M))
    do i = 1, M
        x(i) = 1.0
        y(i) = 1.0
    enddo

!   Vectors rslt_mv, rslt_mv_trans
    allocate(rslt_mv(M))
    allocate(rslt_mv_trans(M))

!   Exit status
    exit_status = 0

!   Create CSR matrices
    info = MKL_SPARSE_D_CREATE_CSR(csrA,SPARSE_INDEX_BASE_ZERO,M,M,csrRowPtr_A(1),csrRowPtr_A(2),csrColInd_A,csrVal_A)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_D_CREATE, csrA: '
        print 100, info
        exit_status = 1
        goto 99
    end if
    info = MKL_SPARSE_D_CREATE_CSR(csrB,SPARSE_INDEX_BASE_ZERO,M,M,csrRowPtr_B(1),csrRowPtr_B(2),csrColInd_B,csrVal_B)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_D_CREATE, csrB: '
        print 100, info
        exit_status = 1
        goto 99
    end if

!   Set hints for A and B
    info = MKL_SPARSE_SET_MV_HINT(csrA, SPARSE_OPERATION_TRANSPOSE, descr_type_gen, 1)
    if ((info .ne. SPARSE_STATUS_SUCCESS) .and. (info .ne. SPARSE_STATUS_NOT_SUPPORTED)) then
        print *, '  MKL_SPARSE_SET_MV_HINT, csrA^t: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    info = MKL_SPARSE_SET_MV_HINT(csrB, SPARSE_OPERATION_TRANSPOSE, descr_type_gen, 1)
    if ((info .ne. SPARSE_STATUS_SUCCESS) .and. (info .ne. SPARSE_STATUS_NOT_SUPPORTED)) then
        print *, '  MKL_SPARSE_SET_MV_HINT, csrB^t: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    info = MKL_SPARSE_SET_MV_HINT(csrB, SPARSE_OPERATION_NON_TRANSPOSE, descr_type_gen, 1)
    if ((info .ne. SPARSE_STATUS_SUCCESS) .and. (info .ne. SPARSE_STATUS_NOT_SUPPORTED)) then
        print *, '  MKL_SPARSE_SET_MV_HINT, csrB: '
        print 100, info
        exit_status = 1
        goto 99
    end if
 
    info = MKL_SPARSE_OPTIMIZE(csrA)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_OPTIMIZE, csrA: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    info = MKL_SPARSE_OPTIMIZE(csrB)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_OPTIMIZE, csrB: '
        print 100, info
        exit_status = 1
        goto 99
    end if

!*****************************************************************************
!   Task 1   
!*****************************************************************************
    print*, '---------------------------------------------------'
    print*, 'Task 1:'
    print*, 'Compute C_1 = A * B using MKL_SPARSE_SP2M'
    
!   Compute C_1 = A*B
    opA = SPARSE_OPERATION_NON_TRANSPOSE
    opB = SPARSE_OPERATION_NON_TRANSPOSE

    request = SPARSE_STAGE_NNZ_COUNT
    info = MKL_SPARSE_SP2M(opA, descr_type_gen, csrA, &
                           opB, descr_type_gen, csrB, &
                           request, csrC_1)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_SP2M with SPARSE_STAGE_NNZ_COUNT, csrC_1: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    request = SPARSE_STAGE_FINALIZE_MULT
    info = MKL_SPARSE_SP2M(opA, descr_type_gen, csrA, &
                           opB, descr_type_gen, csrB, &
                           request, csrC_1)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_SP2M with SPARSE_STAGE_FINALIZE_MULT, csrC_1: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    info = MKL_SPARSE_SET_MV_HINT(csrC_1, SPARSE_OPERATION_NON_TRANSPOSE, descr_type_gen, 1)
    if ((info .ne. SPARSE_STATUS_SUCCESS) .and. (info .ne. SPARSE_STATUS_NOT_SUPPORTED)) then
        print *, '  MKL_SPARSE_SET_MV_HINT, csrC_1: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    info = MKL_SPARSE_OPTIMIZE(csrC_1)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_OPTIMIZE, csrC_1: '
        print 100, info
        exit_status = 1
        goto 99
    end if

!   Executtion routines
!   Step 1:
!        Need to compute the following variables:
!           rslt_mv = C_1 * x
!              left = <rslt_mv, y>

    info = MKL_SPARSE_D_MV(SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrC_1, descr_type_gen, x, beta, rslt_mv)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_D_MV, csrC_1: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    left = DDOT(M, rslt_mv, 1, y, 1)

!   Step 2:
!        Need to compute the following variables:
!           rslt_mv       = B * x
!           rslt_mv_trans = (A^t) * x
!                   right = <rslt_mv, rslt_mv_trans>

    info = MKL_SPARSE_D_MV(SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrB, descr_type_gen, x, beta, rslt_mv)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_D_MV, csrB: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    info = MKL_SPARSE_D_MV(SPARSE_OPERATION_TRANSPOSE, alpha, csrA, descr_type_gen, x, beta, rslt_mv_trans)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_D_MV, csrA^t: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    right = DDOT(M, rslt_mv, 1, rslt_mv_trans, 1)

!   Step 3:
!       Compare values obtained for left and right
    residual_1 = ABS(left - right)/(ABS(left)+1)

    print*,'Check the resultant matrix C_1, using two scalar products.'
    print*,'The difference between < C_1*x , y > and < B*x , (A^t)*y > = ', residual_1, '.'
    if (residual_1 .gt. 1e-8 * DSQRT(DFLOAT(M))) then
        print *, ' Error in Task 1: MKL_SPARSE_SP2M did not arrive at the correct solution.'
        exit_status = 1
        goto 99
    end if
    print*,'It means that MKL_SPARSE_SP2M arrived at the correct solution.'

!   Export CSR matrix
    info = MKL_SPARSE_D_EXPORT_CSR(csrC_1, indexing, nrows, ncols, rows_start_c, rows_end_c, col_indx_c, values_c)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_D_EXPORT_CSR, csrC_1: '
        print 100, info
        exit_status = 1
        goto 99
    end if

!   Converting C into Fortran pointers
    call C_F_POINTER(rows_start_c, rows_start_f, [nrows])
    call C_F_POINTER(rows_end_c  , rows_end_f  , [nrows])
    call C_F_POINTER(col_indx_c  , col_indx_f  , [rows_end_f(nrows)])
    call C_F_POINTER(values_c    , values_f    , [rows_end_f(nrows)])

!   Printing resulting matrix
    print*,'---------------------------------------------------'
    print*,'Output matrix C_1 = A*B:'
    do i = 1, nrows
        print*,'row #',i
        do j = rows_start_f(i)+1, rows_end_f(i)
            print*,col_indx_f(j),values_f(j)
        enddo
    enddo

!*****************************************************************************
!   Task 2   
!*****************************************************************************
    print*, '---------------------------------------------------'
    print*, 'Task 2:'
    print*, 'Compute C_2 = A * (B^t) using MKL_SPARSE_SP2M'
    
!   Compute C_2 = A*(B^t)
    opA = SPARSE_OPERATION_NON_TRANSPOSE
    opB = SPARSE_OPERATION_TRANSPOSE

    request = SPARSE_STAGE_FULL_MULT
    info = MKL_SPARSE_SP2M(opA, descr_type_gen, csrA, &
                           opB, descr_type_gen, csrB, &
                           request, csrC_2)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_SP2M with SPARSE_STAGE_FULL_MULT, csrC_2: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    info = MKL_SPARSE_SET_MV_HINT(csrC_2, SPARSE_OPERATION_NON_TRANSPOSE, descr_type_gen, 1)
    if ((info .ne. SPARSE_STATUS_SUCCESS) .and. (info .ne. SPARSE_STATUS_NOT_SUPPORTED)) then
        print *, '  MKL_SPARSE_SET_MV_HINT, csrC_2: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    info = MKL_SPARSE_OPTIMIZE(csrC_2)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_OPTIMIZE, csrC_2: '
        print 100, info
        exit_status = 1
        goto 99
    end if

!   Executtion routines
!   Step 1:
!        Need to compute the following variables:
!           rslt_mv = C_2 * x
!              left = <rslt_mv, y>

    info = MKL_SPARSE_D_MV(SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrC_2, descr_type_gen, x, beta, rslt_mv)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_D_MV, csrC_2: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    left = DDOT(M, rslt_mv, 1, y, 1)

!   Step 2:
!        Need to compute the following variables:
!           rslt_mv       = (B^t) * x
!           rslt_mv_trans = (A^t) * x
!                   right = <rslt_mv, rslt_mv_trans>

    info = MKL_SPARSE_D_MV(SPARSE_OPERATION_TRANSPOSE, alpha, csrB, descr_type_gen, x, beta, rslt_mv)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_D_MV, csrB^t: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    info = MKL_SPARSE_D_MV(SPARSE_OPERATION_TRANSPOSE, alpha, csrA, descr_type_gen, x, beta, rslt_mv_trans)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_D_MV, csrA^t: '
        print 100, info
        exit_status = 1
        goto 99
    end if

    right = DDOT(M, rslt_mv, 1, rslt_mv_trans, 1)

!   Step 3:
!       Compare values obtained for left and right
    residual_2 = ABS(left - right)/(ABS(left)+1)

    print*,'Check the resultant matrix C_2, using two scalar products.'
    print*,'The difference between < C_2*x , y > and < (B^t)*x , (A^t)*y > = ', residual_2, '.'
    if (residual_2 .gt. 1e-8 * DSQRT(DFLOAT(M))) then
        print *, ' Error in Task 2: MKL_SPARSE_SP2M did not arrive at the correct solution.'
        exit_status = 1
        goto 99
    end if
    print*,'It means that MKL_SPARSE_SP2M arrived at the correct solution.'

!   Export CSR matrix
    info = MKL_SPARSE_D_EXPORT_CSR(csrC_2, indexing, nrows, ncols, rows_start_c, rows_end_c, col_indx_c, values_c)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_D_EXPORT_CSR, csrC_2: '
        print 100, info
        exit_status = 1
        goto 99
    end if

!   Converting C into Fortran pointers
    call C_F_POINTER(rows_start_c, rows_start_f, [nrows])
    call C_F_POINTER(rows_end_c  , rows_end_f  , [nrows])
    call C_F_POINTER(col_indx_c  , col_indx_f  , [rows_end_f(nrows)])
    call C_F_POINTER(values_c    , values_f    , [rows_end_f(nrows)])

!   Printing resulting matrix
    print*,'---------------------------------------------------'
    print*,'Output matrix C_2 = A*(B^t):'
    do i = 1, nrows
        print*,'row #',i
        do j = rows_start_f(i)+1, rows_end_f(i)
            print*,col_indx_f(j),values_f(j)
        enddo
    enddo

 100      format(7x,'ERROR, INFO=',i1)

 99 continue
!   Release internal representation of CSR matrix
    info = MKL_SPARSE_DESTROY(csrC_1)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_DESTROY csrC_1: '
        print 100, info
        exit_status = 1
    end if

    info = MKL_SPARSE_DESTROY(csrC_2)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_DESTROY csrC_2: '
        print 100, info
        exit_status = 1
    end if

    info = MKL_SPARSE_DESTROY(csrA)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_DESTROY csrA: '
        print 100, info
        exit_status = 1
    end if

    info = MKL_SPARSE_DESTROY(csrB)
    if (info .ne. SPARSE_STATUS_SUCCESS) then
        print *, '  MKL_SPARSE_DESTROY csrB: '
        print 100, info
        exit_status = 1
    end if

    print*,'---------------------------------------------------'

    call exit(exit_status)

END PROGRAM SPARSE_SP2M
