/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.progmgr;

import docking.DialogComponentProvider;
import docking.Tool;
import docking.action.DockingAction;
import docking.action.DockingActionIf;
import docking.action.builder.AbstractActionBuilder;
import docking.action.builder.ActionBuilder;
import docking.options.editor.OptionsDialog;
import docking.options.editor.OptionsEditorListener;
import docking.options.editor.StringBasedFileEditor;
import ghidra.app.context.ProgramActionContext;
import ghidra.app.events.CloseProgramPluginEvent;
import ghidra.app.events.ExternalProgramLocationPluginEvent;
import ghidra.app.events.ExternalProgramSelectionPluginEvent;
import ghidra.app.events.OpenProgramPluginEvent;
import ghidra.app.events.ProgramActivatedPluginEvent;
import ghidra.app.events.ProgramClosedPluginEvent;
import ghidra.app.events.ProgramLocationPluginEvent;
import ghidra.app.events.ProgramOpenedPluginEvent;
import ghidra.app.events.ProgramSelectionPluginEvent;
import ghidra.app.plugin.core.progmgr.CloseProgramAction;
import ghidra.app.plugin.core.progmgr.MultiProgramManager;
import ghidra.app.plugin.core.progmgr.ProgramCache;
import ghidra.app.plugin.core.progmgr.ProgramLocator;
import ghidra.app.plugin.core.progmgr.ProgramOptionsAction;
import ghidra.app.plugin.core.progmgr.ProgramSaveManager;
import ghidra.app.plugin.core.progmgr.RedoAction;
import ghidra.app.plugin.core.progmgr.SaveAsProgramAction;
import ghidra.app.plugin.core.progmgr.SaveProgramAction;
import ghidra.app.plugin.core.progmgr.UndoAction;
import ghidra.app.services.ProgramManager;
import ghidra.app.util.NamespaceUtils;
import ghidra.app.util.task.OpenProgramRequest;
import ghidra.app.util.task.OpenProgramTask;
import ghidra.framework.client.ClientUtil;
import ghidra.framework.main.OpenVersionedFileDialog;
import ghidra.framework.model.DomainFile;
import ghidra.framework.model.DomainObject;
import ghidra.framework.model.ProjectData;
import ghidra.framework.model.ProjectLocator;
import ghidra.framework.options.OptionType;
import ghidra.framework.options.Options;
import ghidra.framework.options.OptionsChangeListener;
import ghidra.framework.options.SaveState;
import ghidra.framework.options.ToolOptions;
import ghidra.framework.plugintool.Plugin;
import ghidra.framework.plugintool.PluginEvent;
import ghidra.framework.plugintool.PluginInfo;
import ghidra.framework.plugintool.PluginTool;
import ghidra.framework.plugintool.util.PluginStatus;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressFactory;
import ghidra.program.model.address.AddressRange;
import ghidra.program.model.address.AddressRangeIterator;
import ghidra.program.model.address.AddressSet;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.address.AddressSpace;
import ghidra.program.model.listing.Program;
import ghidra.program.model.symbol.Symbol;
import ghidra.program.model.symbol.SymbolType;
import ghidra.program.util.CodeUnitLocation;
import ghidra.program.util.FunctionSignatureFieldLocation;
import ghidra.program.util.LabelFieldLocation;
import ghidra.program.util.ProgramLocation;
import ghidra.program.util.ProgramSelection;
import ghidra.program.util.ProgramUtilities;
import ghidra.util.HelpLocation;
import ghidra.util.Msg;
import ghidra.util.Swing;
import ghidra.util.exception.AssertException;
import ghidra.util.task.Task;
import ghidra.util.task.TaskLauncher;
import java.awt.Component;
import java.beans.PropertyEditor;
import java.net.MalformedURLException;
import java.net.URL;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;

@PluginInfo(status=PluginStatus.RELEASED, packageName="Ghidra Core", category="Common", shortDescription="Manage open programs", description="This plugin provides actions for opening and closing programs.  It also provides a service to allow plugins to open/close programs.  This plugin is responsible for sending out plugin events to notify all other programs when a program is opened or close.", servicesProvided={ProgramManager.class}, eventsConsumed={OpenProgramPluginEvent.class, CloseProgramPluginEvent.class, ExternalProgramLocationPluginEvent.class, ExternalProgramSelectionPluginEvent.class, ProgramActivatedPluginEvent.class, ProgramLocationPluginEvent.class, ProgramSelectionPluginEvent.class}, eventsProduced={OpenProgramPluginEvent.class, CloseProgramPluginEvent.class, ExternalProgramLocationPluginEvent.class, ExternalProgramSelectionPluginEvent.class, ProgramOpenedPluginEvent.class, ProgramClosedPluginEvent.class, ProgramActivatedPluginEvent.class})
public class ProgramManagerPlugin
extends Plugin
implements ProgramManager,
OptionsChangeListener {
    private static final String CACHE_DURATION_OPTION = "Program Cache.Program Cache Time (minutes)";
    private static final String CACHE_SIZE_OPTION = "Program Cache.Program Cache Size";
    private static final int DEFAULT_PROGRAM_CACHE_CAPACITY = 50;
    private static final int DEFAULT_PROGRAM_CACHE_DURATION = 30;
    private static final String SAVE_GROUP = "DomainObjectSave";
    static final String OPEN_GROUP = "DomainObjectOpen";
    private MultiProgramManager programMgr;
    private ProgramCache programCache;
    private ProgramSaveManager programSaveMgr;
    private int transactionID = -1;
    private UndoAction undoAction;
    private RedoAction redoAction;
    private ProgramLocation currentLocation;

    public ProgramManagerPlugin(PluginTool tool) {
        super(tool);
        this.createActions();
        this.programMgr = new MultiProgramManager(this);
        this.programCache = new ProgramCache(Duration.ofMinutes(30L), 50);
        this.programSaveMgr = new ProgramSaveManager(tool, this);
        this.initializeOptions(tool.getOptions("Tool"));
    }

    public void optionsChanged(ToolOptions options, String optionName, Object oldValue, Object newValue) {
        if (optionName.equals(CACHE_DURATION_OPTION)) {
            Duration duration = Duration.ofMinutes(((Integer)newValue).intValue());
            this.programCache.setDuration(duration);
        }
        if (optionName.equals(CACHE_SIZE_OPTION)) {
            int capacity = (Integer)newValue;
            this.programCache.setCapacity(capacity);
        }
    }

    private void initializeOptions(ToolOptions options) {
        HelpLocation helpLocation = new HelpLocation("Tool", "Program_Cache_Duration");
        options.registerOption(CACHE_DURATION_OPTION, (Object)30, helpLocation, "Sets the time (in minutes) cached programs are kept around before closing");
        helpLocation = new HelpLocation("Tool", "Program_Cache_Size");
        options.registerOption(CACHE_SIZE_OPTION, (Object)50, helpLocation, "Sets the maximum number of programs to be cached");
        int duration = options.getInt(CACHE_DURATION_OPTION, 30);
        int capacity = options.getInt(CACHE_SIZE_OPTION, 50);
        this.programCache.setCapacity(capacity);
        this.programCache.setDuration(Duration.ofMinutes(duration));
        options.addOptionsChangeListener((OptionsChangeListener)this);
    }

    public boolean acceptData(DomainFile[] data) {
        if (data == null || data.length == 0) {
            return false;
        }
        ArrayList<DomainFile> filesToOpen = new ArrayList<DomainFile>();
        for (DomainFile domainFile : data) {
            Class domainObjectClass;
            if (domainFile == null || !Program.class.isAssignableFrom(domainObjectClass = domainFile.getDomainObjectClass())) continue;
            filesToOpen.add(domainFile);
        }
        this.openPrograms(filesToOpen);
        return !filesToOpen.isEmpty();
    }

    public boolean accept(URL url) {
        return this.openProgram(url, 1) != null;
    }

    public Class<?>[] getSupportedDataTypes() {
        return new Class[]{Program.class};
    }

    @Override
    public Program openProgram(DomainFile df) {
        return this.openProgram(df, -1, 1);
    }

    @Override
    public Program openProgram(DomainFile df, int version) {
        return this.openProgram(df, version, 1);
    }

    @Override
    public Program openProgram(DomainFile domainFile, int version, int state) {
        return this.openProgram(new ProgramLocator(domainFile, version), state);
    }

    @Override
    public Program openProgram(URL ghidraURL, int state) {
        String location = ghidraURL.getRef();
        Program program = this.openProgram(new ProgramLocator(ghidraURL), state);
        if (program != null && location != null && state == 1) {
            this.gotoProgramRef(program, ghidraURL.getRef());
            this.programMgr.saveLocation();
        }
        return program;
    }

    private Program openProgram(ProgramLocator locator, int state) {
        Program program = (Program)Swing.runNow(() -> this.doOpenProgramSwing(locator, state));
        if (program != null) {
            Msg.info((Object)this, (Object)("Opened program in " + this.tool.getName() + " tool: " + String.valueOf(locator)));
        }
        return program;
    }

    private Program doOpenProgramSwing(ProgramLocator locator, int state) {
        Program program = this.programMgr.getOpenProgram(locator);
        if (program != null) {
            this.showProgram(program, locator, state);
            return program;
        }
        program = (Program)this.programCache.get(locator);
        if (program != null) {
            this.programMgr.addProgram(program, locator, state);
            return program;
        }
        OpenProgramTask task = new OpenProgramTask(locator, (Object)this);
        new TaskLauncher((Task)task, (Component)this.tool.getToolFrame());
        OpenProgramRequest request = task.getOpenProgram();
        if (request != null) {
            program = request.getProgram();
            this.programMgr.addProgram(program, locator, state);
            request.release();
            return program;
        }
        return null;
    }

    private boolean gotoProgramRef(Program program, String ref) {
        if (StringUtils.isBlank((CharSequence)ref)) {
            return false;
        }
        String trimmedRef = ref.trim();
        ProgramLocation loc = this.getLocationForSymbolRef(program, trimmedRef);
        if (loc == null) {
            loc = this.getLocationForAddressRef(program, trimmedRef);
        }
        if (loc == null) {
            Msg.showError((Object)this, null, (String)"Navigation Failed", (Object)("Referenced label/function not found: " + trimmedRef));
            return false;
        }
        this.firePluginEvent(new ProgramLocationPluginEvent(this.getName(), loc, program));
        return true;
    }

    private ProgramLocation getLocationForAddressRef(Program program, String ref) {
        Address addr = program.getAddressFactory().getAddress(ref);
        if (addr != null && addr.isMemoryAddress()) {
            return new CodeUnitLocation(program, addr, 0, 0, 0);
        }
        return null;
    }

    private ProgramLocation getLocationForSymbolRef(Program program, String ref) {
        List symbols = NamespaceUtils.getSymbols((String)ref, (Program)program);
        if (symbols.isEmpty()) {
            return null;
        }
        Symbol symbol = (Symbol)symbols.get(0);
        if (symbol == null) {
            return null;
        }
        SymbolType type = symbol.getSymbolType();
        if (type == SymbolType.FUNCTION) {
            return new FunctionSignatureFieldLocation(program, symbol.getAddress());
        }
        if (type == SymbolType.LABEL) {
            return new LabelFieldLocation(symbol);
        }
        return null;
    }

    @Override
    public Program openCachedProgram(URL ghidraURL, Object consumer) {
        return this.openCachedProgram(new ProgramLocator(ghidraURL), consumer);
    }

    @Override
    public Program openCachedProgram(DomainFile domainFile, Object consumer) {
        return this.openCachedProgram(new ProgramLocator(domainFile), consumer);
    }

    private Program openCachedProgram(ProgramLocator locator, Object consumer) {
        Program program = (Program)this.programCache.get(locator);
        if (program != null) {
            program.addConsumer(consumer);
            return program;
        }
        program = this.programMgr.getOpenProgram(locator);
        if (program != null) {
            program.addConsumer(consumer);
            if (!program.isChanged() || ProgramUtilities.isChangedWithUpgradeOnly((Program)program)) {
                this.programCache.put(locator, program);
            }
            return program;
        }
        OpenProgramTask task = new OpenProgramTask(locator, consumer);
        new TaskLauncher((Task)task, (Component)this.tool.getToolFrame());
        OpenProgramRequest result = task.getOpenProgram();
        if (result == null) {
            return null;
        }
        program = result.getProgram();
        this.programCache.put(locator, program);
        return program;
    }

    @Override
    public Program getCurrentProgram() {
        return this.programMgr.getCurrentProgram();
    }

    public DomainFile[] getData() {
        Program[] p = this.getAllOpenPrograms();
        DomainFile[] dfs = new DomainFile[p.length];
        for (int i = 0; i < dfs.length; ++i) {
            dfs[i] = p[i].getDomainFile();
        }
        return dfs;
    }

    @Override
    public Program[] getAllOpenPrograms() {
        return (Program[])this.programMgr.getAllPrograms().toArray(Program[]::new);
    }

    public void dispose() {
        this.programCache.clear();
        this.programMgr.dispose();
        this.tool.clearLastEvents();
    }

    @Override
    public boolean closeOtherPrograms(boolean ignoreChanges) {
        List<Program> otherPrograms = this.programMgr.getOtherPrograms();
        Runnable r = () -> this.doCloseAllPrograms(otherPrograms, ignoreChanges);
        Swing.runNow((Runnable)r);
        return this.programMgr.isEmpty();
    }

    @Override
    public boolean closeAllPrograms(boolean ignoreChanges) {
        List<Program> openPrograms = this.programMgr.getAllPrograms();
        Runnable r = () -> this.doCloseAllPrograms(openPrograms, ignoreChanges);
        Swing.runNow((Runnable)r);
        return this.programMgr.isEmpty();
    }

    private void doCloseAllPrograms(List<Program> openPrograms, boolean ignoreChanges) {
        ArrayList<Program> toRemove = new ArrayList<Program>();
        Program currentProgram = this.programMgr.getCurrentProgram();
        for (Program p : openPrograms) {
            if (ignoreChanges) {
                toRemove.add(p);
                continue;
            }
            if (p.isClosed()) {
                toRemove.add(p);
                continue;
            }
            if (!this.tool.canCloseDomainObject((DomainObject)p)) continue;
            if (!this.programSaveMgr.canClose(p)) {
                return;
            }
            toRemove.add(p);
        }
        if (toRemove.contains(currentProgram)) {
            toRemove.remove(currentProgram);
            toRemove.add(currentProgram);
        }
        for (Program program : toRemove) {
            this.programMgr.removeProgram(program);
        }
        this.contextChanged();
    }

    @Override
    public boolean closeProgram(Program program, boolean ignoreChanges) {
        if (program == null) {
            return false;
        }
        Runnable r = () -> {
            if (ignoreChanges || program.isClosed() || this.programMgr.isPersistent(program) || this.tool.canCloseDomainObject((DomainObject)program) && this.programSaveMgr.canClose(program)) {
                this.programMgr.removeProgram(program);
                this.contextChanged();
            }
        };
        Swing.runNow((Runnable)r);
        return !this.programMgr.contains(program);
    }

    protected void close() {
        List<Program> programs = this.programMgr.getAllPrograms();
        if (programs.isEmpty()) {
            return;
        }
        Program currentProgram = this.getCurrentProgram();
        for (Program program : programs) {
            if (program == currentProgram) continue;
            this.programMgr.removeProgram(program);
        }
        if (currentProgram != null) {
            this.programMgr.removeProgram(currentProgram);
        }
        this.contextChanged();
        this.tool.setSubTitle("");
        this.tool.clearLastEvents();
    }

    @Override
    public void setCurrentProgram(Program p) {
        Runnable r = () -> {
            this.programMgr.setCurrentProgram(p);
            this.contextChanged();
        };
        Swing.runNow((Runnable)r);
    }

    @Override
    public Program getProgram(Address addr) {
        return this.programMgr.getProgram(addr);
    }

    @Override
    public void openProgram(Program program) {
        this.openProgram(program, 1);
    }

    @Override
    public void openProgram(Program program, int state) {
        this.showProgram(program, new ProgramLocator(program.getDomainFile()), state);
    }

    private void showProgram(Program p, ProgramLocator locator, int state) {
        if (p == null || p.isClosed()) {
            throw new AssertException("Opened program required");
        }
        Runnable r = () -> {
            this.programMgr.addProgram(p, locator, state);
            if (state == 1) {
                this.programMgr.saveLocation();
            }
            this.contextChanged();
        };
        Swing.runNow((Runnable)r);
    }

    @Override
    public boolean closeProgram() {
        return this.closeProgram(this.getCurrentProgram(), false);
    }

    protected boolean saveData() {
        boolean result = this.programSaveMgr.saveAll();
        this.contextChanged();
        return result;
    }

    protected boolean hasUnsaveData() {
        Program[] allOpenPrograms;
        for (Program program : allOpenPrograms = this.getAllOpenPrograms()) {
            if (!program.isChanged()) continue;
            return true;
        }
        return false;
    }

    private void createActions() {
        int subMenuGroup = 1;
        DockingAction openAction = (DockingAction)((ActionBuilder)((ActionBuilder)((ActionBuilder)((ActionBuilder)new ActionBuilder("Open File", this.getName()).menuPath(new String[]{"&File", "&Open..."})).menuGroup(OPEN_GROUP, Integer.toString(subMenuGroup++))).keyBinding("ctrl O")).onAction(c -> this.open())).buildAndInstall((Tool)this.tool);
        openAction.addToWindowWhen(ProgramActionContext.class);
        this.tool.addAction((DockingActionIf)new CloseProgramAction(this, OPEN_GROUP, subMenuGroup++));
        ((ActionBuilder)((ActionBuilder)((ActionBuilder)new ActionBuilder("Close Others", this.getName()).menuPath(new String[]{"&File", "Close &Others"})).menuGroup(OPEN_GROUP, Integer.toString(subMenuGroup++))).enabled(false)).withContext(ProgramActionContext.class).inWindow(AbstractActionBuilder.When.CONTEXT_MATCHES).enabledWhen(c -> this.programMgr.contains(c.getProgram())).onAction(c -> this.closeOtherPrograms(false)).buildAndInstall((Tool)this.tool);
        DockingAction closeAllAction = (DockingAction)((ActionBuilder)((ActionBuilder)((ActionBuilder)((ActionBuilder)new ActionBuilder("Close All", this.getName()).menuPath(new String[]{"&File", "Close &All"})).menuGroup(OPEN_GROUP, Integer.toString(subMenuGroup++))).enabledWhen(c -> !this.programMgr.isEmpty())).onAction(c -> this.closeAllPrograms(false))).buildAndInstall((Tool)this.tool);
        closeAllAction.addToWindowWhen(ProgramActionContext.class);
        this.tool.addAction((DockingActionIf)new SaveProgramAction(this, SAVE_GROUP, subMenuGroup));
        this.tool.addAction((DockingActionIf)new SaveAsProgramAction(this, SAVE_GROUP, subMenuGroup));
        DockingAction saveAllAction = (DockingAction)((ActionBuilder)((ActionBuilder)((ActionBuilder)((ActionBuilder)((ActionBuilder)new ActionBuilder("Save All Files", this.getName()).menuPath(new String[]{"&File", "Save All"})).description("Save All Programs")).menuGroup(SAVE_GROUP, Integer.toString(subMenuGroup++))).enabledWhen(c -> this.programMgr.hasUnsavedPrograms())).onAction(c -> this.programSaveMgr.saveChangedPrograms())).buildAndInstall((Tool)this.tool);
        saveAllAction.addToWindowWhen(ProgramActionContext.class);
        this.tool.addAction((DockingActionIf)new ProgramOptionsAction(this));
        this.undoAction = new UndoAction(this, this.tool);
        this.redoAction = new RedoAction(this, this.tool);
        this.tool.addAction((DockingActionIf)this.undoAction);
        this.tool.addAction((DockingActionIf)this.redoAction);
    }

    void showProgramOptions(final Program currentProgram) {
        List names = currentProgram.getOptionsNames();
        Options[] options = new Options[names.size()];
        for (int i = 0; i < names.size(); ++i) {
            String optionName = (String)names.get(i);
            options[i] = currentProgram.getOptions(optionName);
            if (!optionName.equals("Program Information")) continue;
            this.setPropertyEditor(options[i], "Executable Location");
            options[i].setOptionsHelpLocation(new HelpLocation(this.getName(), "Program_Options"));
        }
        OptionsDialog dialog = new OptionsDialog("Properties for " + currentProgram.getName(), "Properties", options, new OptionsEditorListener(){

            public void beforeChangesApplied() {
                ProgramManagerPlugin.this.startTransaction(currentProgram);
            }

            public void changesApplied() {
                ProgramManagerPlugin.this.endTransaction(currentProgram);
            }
        });
        dialog.setHelpLocation(new HelpLocation("ProgramManagerPlugin", "Program_Options"));
        this.tool.showDialog((DialogComponentProvider)dialog);
        dialog.dispose();
    }

    private void setPropertyEditor(Options options, String filePropertyName) {
        PropertyEditor editor = options.getPropertyEditor(filePropertyName);
        if (editor == null && options.getType(filePropertyName) == OptionType.STRING_TYPE) {
            options.registerOption(filePropertyName, OptionType.STRING_TYPE, null, null, null, () -> new StringBasedFileEditor());
        }
    }

    private void startTransaction(Program currentProgram) {
        if (this.transactionID < 0) {
            this.transactionID = currentProgram.startTransaction("Edit Program Properties");
        }
    }

    private void endTransaction(Program currentProgram) {
        if (this.transactionID >= 0) {
            currentProgram.endTransaction(this.transactionID, true);
            this.transactionID = -1;
        }
    }

    void contextChanged() {
        this.tool.contextChanged(null);
    }

    private void open() {
        OpenVersionedFileDialog<Program> openDialog = new OpenVersionedFileDialog<Program>(this.tool, "Open Program", Program.class);
        DomainFile startFile = null;
        Program p = this.getCurrentProgram();
        if (p != null) {
            startFile = p.getDomainFile();
        }
        openDialog.selectDomainFile(startFile);
        openDialog.setHelpLocation(new HelpLocation("ProgramManagerPlugin", "Open_File_Dialog"));
        openDialog.addOkActionListener(e -> {
            DomainFile domainFile = openDialog.getDomainFile();
            int version = openDialog.getVersion();
            if (domainFile == null) {
                openDialog.setStatusText("Please choose a Program");
            } else {
                openDialog.close();
                this.doOpenProgramSwing(new ProgramLocator(domainFile, version), 1);
            }
        });
        this.tool.showDialog(openDialog);
        this.contextChanged();
    }

    public void openPrograms(List<DomainFile> filesToOpen) {
        List<ProgramLocator> locators = filesToOpen.stream().map(f -> new ProgramLocator((DomainFile)f)).collect(Collectors.toList());
        this.openProgramLocations(locators);
    }

    private void openProgramLocations(List<ProgramLocator> locators) {
        if (locators.isEmpty()) {
            return;
        }
        LinkedHashSet<ProgramLocator> toOpen = new LinkedHashSet<ProgramLocator>(locators);
        Map<ProgramLocator, Program> alreadyOpen = this.getOpenPrograms(toOpen);
        this.makeVisibleInTool(alreadyOpen.values());
        toOpen.removeAll(alreadyOpen.keySet());
        Map<ProgramLocator, Program> openedFromCache = this.openCachedProgramsInTool(toOpen);
        toOpen.removeAll(openedFromCache.keySet());
        if (toOpen.isEmpty()) {
            Program first = this.programMgr.getOpenProgram(locators.get(0));
            this.showProgram(first, locators.get(0), 1);
            return;
        }
        OpenProgramTask task = new OpenProgramTask(new ArrayList<ProgramLocator>(toOpen), (Object)this);
        new TaskLauncher((Task)task, (Component)this.tool.getToolFrame());
        List<OpenProgramRequest> openProgramReqs = task.getOpenPrograms();
        int openState = 1;
        for (OpenProgramRequest programReq : openProgramReqs) {
            this.showProgram(programReq.getProgram(), programReq.getLocator(), openState);
            programReq.release();
            openState = 2;
        }
    }

    private void makeVisibleInTool(Collection<Program> programs) {
        for (Program program : programs) {
            this.openProgram(program, 2);
        }
    }

    private Map<ProgramLocator, Program> openCachedProgramsInTool(Set<ProgramLocator> toOpen) {
        HashMap<ProgramLocator, Program> map = new HashMap<ProgramLocator, Program>();
        for (ProgramLocator programLocator : toOpen) {
            Program program = (Program)this.programCache.get(programLocator);
            if (program == null) continue;
            this.openProgram(program, 2);
            map.put(programLocator, program);
        }
        return map;
    }

    private Map<ProgramLocator, Program> getOpenPrograms(Collection<ProgramLocator> locators) {
        HashMap<ProgramLocator, Program> map = new HashMap<ProgramLocator, Program>();
        for (ProgramLocator locator : locators) {
            Program program = this.programMgr.getOpenProgram(locator);
            if (program == null) continue;
            map.put(locator, program);
        }
        return map;
    }

    @Override
    public void saveProgram() {
        this.saveProgram(this.getCurrentProgram());
    }

    @Override
    public void saveProgram(Program program) {
        Swing.runIfSwingOrRunLater(() -> this.programSaveMgr.saveProgram(program));
    }

    @Override
    public void saveProgramAs() {
        this.saveProgramAs(this.getCurrentProgram());
    }

    @Override
    public void saveProgramAs(Program program) {
        Swing.runIfSwingOrRunLater(() -> this.programSaveMgr.saveAs(program));
    }

    public void writeDataState(SaveState saveState) {
        ProjectLocator projectLocator;
        ArrayList<MultiProgramManager.ProgramInfo> programInfos = new ArrayList<MultiProgramManager.ProgramInfo>();
        for (Program program : this.programMgr.getAllPrograms()) {
            MultiProgramManager.ProgramInfo info = this.programMgr.getInfo(program);
            if (info == null || !info.canReopen()) continue;
            programInfos.add(info);
        }
        saveState.putInt("NUM_PROGRAMS", programInfos.size());
        int i = 0;
        for (MultiProgramManager.ProgramInfo programInfo : programInfos) {
            this.writeProgramInfo(programInfo, saveState, i++);
        }
        Program program = this.programMgr.getCurrentProgram();
        if (program != null && (projectLocator = program.getDomainFile().getProjectLocator()) != null && !projectLocator.isTransient()) {
            saveState.putString("CURRENT_FILE", program.getDomainFile().getName());
            if (this.currentLocation != null) {
                this.currentLocation.saveState(saveState);
            }
        }
    }

    public void readDataState(SaveState saveState) {
        if (!this.programMgr.isEmpty()) {
            this.currentLocation = null;
            return;
        }
        this.loadPrograms(saveState);
        String currentFile = saveState.getString("CURRENT_FILE", null);
        List<Program> programs = this.programMgr.getAllPrograms();
        if (!programs.isEmpty()) {
            if (currentFile != null) {
                for (Program program : programs) {
                    if (!program.getDomainFile().getName().equals(currentFile)) continue;
                    this.programMgr.setCurrentProgram(program);
                    this.currentLocation = ProgramLocation.getLocation((Program)program, (SaveState)saveState);
                    break;
                }
            }
            if (this.getCurrentProgram() == null) {
                this.programMgr.setCurrentProgram(programs.get(0));
            }
        }
        this.contextChanged();
    }

    public void dataStateRestoreCompleted() {
        if (this.currentLocation != null) {
            this.tool.firePluginEvent((PluginEvent)new ProgramLocationPluginEvent(this.getName(), this.currentLocation, this.getCurrentProgram()));
        }
    }

    private void writeProgramInfo(MultiProgramManager.ProgramInfo programInfo, SaveState saveState, int index) {
        if (programInfo.getProgramLocator().isURL()) {
            URL url = programInfo.getProgramLocator().getURL();
            saveState.putString("URL_" + index, url.toString());
            return;
        }
        String projectLocation = null;
        String projectName = null;
        String path = null;
        Program program = programInfo.program;
        DomainFile df = program.getDomainFile();
        ProjectLocator projectLocator = df.getProjectLocator();
        if (projectLocator != null && !projectLocator.isTransient()) {
            projectLocation = projectLocator.getLocation();
            projectName = projectLocator.getName();
            path = df.getPathname();
        }
        int version = -1;
        if (!df.isLatestVersion()) {
            version = df.getVersion();
        }
        saveState.putString("LOCATION_" + index, projectLocation);
        saveState.putString("PROJECT_NAME_" + index, projectName);
        saveState.putInt("VERSION_" + index, version);
        saveState.putString("PATHNAME_" + index, path);
    }

    private void loadPrograms(SaveState saveState) {
        int programCount = saveState.getInt("NUM_PROGRAMS", 0);
        if (programCount == 0) {
            return;
        }
        ArrayList<ProgramLocator> openList = new ArrayList<ProgramLocator>();
        for (int index = 0; index < programCount; ++index) {
            URL url = this.getGhidraURL(saveState, index);
            if (url != null) {
                openList.add(new ProgramLocator(url));
                continue;
            }
            DomainFile domainFile = this.getDomainFile(saveState, index);
            if (domainFile == null) continue;
            int version = this.getVersion(saveState, index);
            openList.add(new ProgramLocator(domainFile, version));
        }
        if (openList.isEmpty()) {
            return;
        }
        OpenProgramTask task = new OpenProgramTask(openList, (Object)this);
        task.setNoCheckout();
        new TaskLauncher((Task)task, (Component)this.tool.getToolFrame(), 100);
        List<OpenProgramRequest> openProgramReqs = task.getOpenPrograms();
        for (OpenProgramRequest programReq : openProgramReqs) {
            ProgramLocator locator = programReq.getLocator();
            this.showProgram(programReq.getProgram(), locator, 2);
            programReq.release();
        }
    }

    private URL getGhidraURL(SaveState saveState, int index) {
        String url = saveState.getString("URL_" + index, null);
        if (url == null) {
            return null;
        }
        try {
            return new URL(url);
        }
        catch (MalformedURLException e) {
            return null;
        }
    }

    private DomainFile getDomainFile(SaveState saveState, int index) {
        String pathname = saveState.getString("PATHNAME_" + index, null);
        String location = saveState.getString("LOCATION_" + index, null);
        String projectName = saveState.getString("PROJECT_NAME_" + index, null);
        if (location == null || projectName == null) {
            return null;
        }
        ProjectLocator projectLocator = new ProjectLocator(location, projectName);
        ProjectData projectData = this.tool.getProject().getProjectData(projectLocator);
        if (projectData == null) {
            return null;
        }
        DomainFile df = projectData.getFile(pathname);
        if (df == null) {
            String message = "Can't open program - \"" + pathname + "\"";
            int version = this.getVersion(saveState, index);
            if (version != -1) {
                message = message + " version " + version;
            }
            Msg.showError((Object)this, (Component)this.tool.getToolFrame(), (String)"Program Not Found", (Object)message);
        }
        return df;
    }

    private int getVersion(SaveState saveState, int index) {
        return saveState.getInt("VERSION_" + index, -1);
    }

    public void processEvent(PluginEvent event) {
        if (event instanceof OpenProgramPluginEvent) {
            OpenProgramPluginEvent ev = (OpenProgramPluginEvent)event;
            this.openProgram(ev.getProgram());
        } else if (event instanceof CloseProgramPluginEvent) {
            CloseProgramPluginEvent ev = (CloseProgramPluginEvent)event;
            this.closeProgram(ev.getProgram(), ev.ignoreChanges());
        } else if (event instanceof ProgramActivatedPluginEvent) {
            Program p = ((ProgramActivatedPluginEvent)event).getActiveProgram();
            this.programMgr.setCurrentProgram(p);
        } else if (event instanceof ProgramLocationPluginEvent) {
            ProgramLocationPluginEvent ev = (ProgramLocationPluginEvent)event;
            this.currentLocation = ev.getLocation();
            this.firePluginEvent(new ExternalProgramLocationPluginEvent(this.getName(), this.currentLocation, ev.getProgram()));
        } else if (event instanceof ExternalProgramLocationPluginEvent) {
            Program currentProgram = this.programMgr.getCurrentProgram();
            if (currentProgram == null) {
                return;
            }
            ExternalProgramLocationPluginEvent ev = (ExternalProgramLocationPluginEvent)event;
            ProgramLocation loc = ev.getLocation();
            if (loc != null && (loc = this.localizeLocation(currentProgram, loc)) != null && currentProgram.getAddressFactory().isValidAddress(loc.getAddress()) && currentProgram.getMemory().contains(loc.getAddress())) {
                this.firePluginEvent(new ProgramLocationPluginEvent(this.getName(), loc, ev.getProgram()));
            }
        } else if (event instanceof ProgramSelectionPluginEvent) {
            ProgramSelectionPluginEvent ev = (ProgramSelectionPluginEvent)event;
            this.firePluginEvent(new ExternalProgramSelectionPluginEvent(this.getName(), ev.getSelection(), ev.getProgram()));
        } else if (event instanceof ExternalProgramSelectionPluginEvent) {
            Program currentProgram = this.programMgr.getCurrentProgram();
            ExternalProgramSelectionPluginEvent ev = (ExternalProgramSelectionPluginEvent)event;
            if (currentProgram == null) {
                return;
            }
            ProgramSelection sel = ev.getSelection();
            if (sel != null && this.hasValidAddresses(currentProgram, sel = this.localizeSelection(currentProgram, sel))) {
                this.firePluginEvent(new ProgramSelectionPluginEvent(this.getName(), sel, ev.getProgram()));
            }
        }
    }

    private ProgramSelection localizeSelection(Program currentProgram, ProgramSelection sel) {
        if (this.hasValidAddresses(currentProgram, sel)) {
            return sel;
        }
        AddressFactory addrFactory = currentProgram.getAddressFactory();
        AddressSpace defaultSpace = addrFactory.getDefaultAddressSpace();
        AddressSet locAddressSet = new AddressSet();
        AddressRangeIterator riter = sel.getAddressRanges();
        while (riter.hasNext()) {
            AddressRange range = (AddressRange)riter.next();
            Address min = range.getMinAddress();
            Address max = range.getMaxAddress();
            try {
                min = defaultSpace.getAddress(min.getOffset());
                max = defaultSpace.getAddress(max.getOffset());
                locAddressSet.addRange(min, max);
            }
            catch (Exception exception) {}
        }
        return new ProgramSelection((AddressSetView)locAddressSet);
    }

    private ProgramLocation localizeLocation(Program currentProgram, ProgramLocation loc) {
        Address addr = loc.getAddress();
        Address refAddr = loc.getRefAddress();
        if (loc.isValid(currentProgram)) {
            return loc;
        }
        AddressFactory addressFactory = currentProgram.getAddressFactory();
        try {
            addr = addressFactory.getAddress(addr.toString(true));
            if (addr == null) {
                return null;
            }
        }
        catch (Exception e) {
            return null;
        }
        if (refAddr != null) {
            try {
                refAddr = addressFactory.getAddress(refAddr.toString(true));
            }
            catch (Exception e) {
                refAddr = null;
            }
        }
        return new ProgramLocation(currentProgram, addr, loc.getComponentPath(), refAddr, 0, 0, 0);
    }

    private boolean hasValidAddresses(Program currentProgram, ProgramSelection sel) {
        AddressRangeIterator it = sel.getAddressRanges();
        AddressFactory af = currentProgram.getAddressFactory();
        while (it.hasNext()) {
            AddressRange range = (AddressRange)it.next();
            if (af.isValidAddress(range.getMinAddress())) continue;
            return false;
        }
        return true;
    }

    @Override
    public boolean isVisible(Program program) {
        return this.programMgr.isVisible(program);
    }

    @Override
    public void releaseProgram(Program program, Object owner) {
        if (this.programMgr.contains(program)) {
            this.programMgr.releaseProgram(program, owner);
            Msg.info(ClientUtil.class, (Object)("Released program from " + this.tool.getName() + " tool: " + String.valueOf(program.getDomainFile())));
        }
    }

    @Override
    public boolean setPersistentOwner(Program program, Object owner) {
        return this.programMgr.setPersistentOwner(program, owner);
    }

    public boolean isManaged(Program program) {
        return this.programMgr.contains(program);
    }
}

