// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
/*
 * This public interface for serializing the AWS Encryption SDK Message Header Format
 * is provided for the use of the Encryption SDK for JavaScript only.  It can be used
 * as a reference but is not intended to be use by any packages other than the
 * Encryption SDK for JavaScript.
 *
 * See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-structure
 */
import { concatBuffers } from './concat_buffers';
import { needs, MessageFormat, } from '@aws-crypto/material-management';
import { ContentType, ObjectType, SequenceIdentifier, SerializationVersion, } from './identifiers';
import { uInt16BE, uInt8, uInt32BE } from './uint_util';
export function serializeFactory(fromUtf8) {
    return {
        frameIv,
        nonFramedBodyIv,
        headerAuthIv,
        frameHeader,
        finalFrameHeader,
        encodeEncryptionContext,
        serializeEncryptionContext,
        serializeEncryptedDataKeys,
        serializeEncryptedDataKey,
        serializeMessageHeader,
        buildMessageHeader,
    };
    function frameIv(ivLength, sequenceNumber) {
        /* Precondition: sequenceNumber must conform to the specification. i.e. 1 - (2^32 - 1)
         * The sequence number starts at 1
         * https://github.com/awslabs/aws-encryption-sdk-specification/blob/master/data-format/message-body.md#sequence-number
         */
        needs(sequenceNumber > 0 &&
            SequenceIdentifier.SEQUENCE_NUMBER_END >= sequenceNumber, 'sequenceNumber out of bounds');
        const buff = new Uint8Array(ivLength);
        const view = new DataView(buff.buffer, buff.byteOffset, buff.byteLength);
        view.setUint32(ivLength - 4, sequenceNumber, false); // big-endian
        return buff;
    }
    function nonFramedBodyIv(ivLength) {
        return frameIv(ivLength, 1);
    }
    function headerAuthIv(ivLength) {
        return new Uint8Array(ivLength); // new Uint8Array is 0 filled by default
    }
    function frameHeader(sequenceNumber, iv) {
        return concatBuffers(uInt32BE(sequenceNumber), iv);
    }
    function finalFrameHeader(sequenceNumber, iv, contentLength) {
        return concatBuffers(uInt32BE(SequenceIdentifier.SEQUENCE_NUMBER_END), // Final Frame identifier
        uInt32BE(sequenceNumber), iv, uInt32BE(contentLength));
    }
    function encodeEncryptionContext(encryptionContext) {
        return (Object.entries(encryptionContext)
            /* Precondition: The serialized encryption context entries must be sorted by UTF-8 key value. */
            .sort(([aKey], [bKey]) => aKey.localeCompare(bKey))
            .map((entries) => entries.map(fromUtf8))
            .map(([key, value]) => concatBuffers(uInt16BE(key.byteLength), key, uInt16BE(value.byteLength), value)));
    }
    function serializeEncryptionContext(encryptionContext) {
        const encryptionContextElements = encodeEncryptionContext(encryptionContext);
        /* Check for early return (Postcondition): If there is no context then the length of the _whole_ serialized portion is 0.
         * This is part of the specification of the AWS Encryption SDK Message Format.
         * It is not 0 for length and 0 for count.  The count element is omitted.
         */
        if (!encryptionContextElements.length)
            return uInt16BE(0);
        const aadData = concatBuffers(uInt16BE(encryptionContextElements.length), ...encryptionContextElements);
        const aadLength = uInt16BE(aadData.byteLength);
        return concatBuffers(aadLength, aadData);
    }
    function serializeEncryptedDataKeys(encryptedDataKeys) {
        const encryptedKeyInfo = encryptedDataKeys.map(serializeEncryptedDataKey);
        return concatBuffers(uInt16BE(encryptedDataKeys.length), ...encryptedKeyInfo);
    }
    function serializeEncryptedDataKey(edk) {
        const { providerId, providerInfo, encryptedDataKey, rawInfo } = edk;
        const providerIdBytes = fromUtf8(providerId);
        // The providerInfo is technically a binary field, so I prefer rawInfo
        const providerInfoBytes = rawInfo || fromUtf8(providerInfo);
        return concatBuffers(uInt16BE(providerIdBytes.byteLength), providerIdBytes, uInt16BE(providerInfoBytes.byteLength), providerInfoBytes, uInt16BE(encryptedDataKey.byteLength), encryptedDataKey);
    }
    function serializeMessageHeader(messageHeader) {
        /* Precondition: Must be a version that can be serialized. */
        needs(SerializationVersion[messageHeader.version], 'Unsupported version.');
        if (messageHeader.version === 1) {
            return serializeMessageHeaderV1(messageHeader);
        }
        else {
            return serializeMessageHeaderV2(messageHeader);
        }
    }
    function serializeMessageHeaderV1(messageHeader) {
        return concatBuffers(uInt8(messageHeader.version), uInt8(messageHeader.type), uInt16BE(messageHeader.suiteId), messageHeader.messageId, serializeEncryptionContext(messageHeader.encryptionContext), serializeEncryptedDataKeys(messageHeader.encryptedDataKeys), new Uint8Array([messageHeader.contentType]), new Uint8Array([0, 0, 0, 0]), uInt8(messageHeader.headerIvLength), uInt32BE(messageHeader.frameLength));
    }
    function serializeMessageHeaderV2(messageHeader) {
        return concatBuffers(uInt8(messageHeader.version), uInt16BE(messageHeader.suiteId), messageHeader.messageId, serializeEncryptionContext(messageHeader.encryptionContext), serializeEncryptedDataKeys(messageHeader.encryptedDataKeys), new Uint8Array([messageHeader.contentType]), uInt32BE(messageHeader.frameLength), messageHeader.suiteData);
    }
    /* This _could_ take the material directly.
     * But I don't do that on purpose.
     * It may be overly paranoid,
     * but this way once the material is created,
     * it has a minimum of egress.
     */
    function buildMessageHeader({ encryptionContext, encryptedDataKeys, suite, messageId, frameLength, suiteData, }) {
        const { messageFormat: version, id: suiteId } = suite;
        const contentType = ContentType.FRAMED_DATA;
        if (version === MessageFormat.V1) {
            const type = ObjectType.CUSTOMER_AE_DATA;
            const { ivLength: headerIvLength } = suite;
            return {
                version,
                type,
                suiteId,
                messageId,
                encryptionContext,
                encryptedDataKeys,
                contentType,
                headerIvLength,
                frameLength,
            };
        }
        else if (version === MessageFormat.V2) {
            return {
                version,
                suiteId,
                messageId,
                encryptionContext: encryptionContext,
                encryptedDataKeys: encryptedDataKeys,
                contentType,
                frameLength,
                suiteData,
            };
        }
        needs(false, 'Unsupported message format version.');
    }
}
export function serializeMessageHeaderAuth({ headerIv, headerAuthTag, messageHeader, }) {
    if (messageHeader.version === MessageFormat.V1) {
        return concatBuffers(headerIv, headerAuthTag);
    }
    return headerAuthTag;
}
//# sourceMappingURL=data:application/json;base64,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