import { CodeInterpreterTool } from '../assistant';
import { ChatCompletionMessage, ChatCompletionRole } from '../inference';
import { ModelInfo } from '../model';
import { Thread } from '../thread';
/**
 * The `ThreadMessage` type defines the shape of a thread's message object.
 * @stored
 */
export type ThreadMessage = {
    /** Unique identifier for the message, generated by default using the ULID method. **/
    id: string;
    /** Object name **/
    object: string;
    /** Thread id, default is a ulid. **/
    thread_id: string;
    /** The assistant id of this thread. **/
    assistant_id?: string;
    /**
     * A list of files attached to the message, and the tools they were added to.
     */
    attachments?: Array<Attachment> | null;
    /** The role of the author of this message. **/
    role: ChatCompletionRole;
    /** The content of this message. **/
    content: ThreadContent[];
    /** The status of this message. **/
    status: MessageStatus;
    /** The timestamp indicating when this message was created. Represented in Unix time. **/
    created_at: number;
    /** The timestamp indicating when this message was updated. Represented in Unix time. **/
    completed_at: number;
    /** The additional metadata of this message. **/
    metadata?: Record<string, unknown>;
    /** Type of the message */
    type?: string;
    /** The error code which explain what error type. Used in conjunction with MessageStatus.Error */
    error_code?: ErrorCode;
    tool_call_id?: string;
};
/**
 * The `MessageRequest` type defines the shape of a new message request object.
 * @data_transfer_object
 */
export type MessageRequest = {
    /**
     * The id of the message request.
     */
    id?: string;
    /**
     * @deprecated Use thread object instead
     * The thread id of the message request.
     */
    threadId: string;
    /**
     * The assistant id of the message request.
     */
    assistantId?: string;
    /**
     * A list of files attached to the message, and the tools they were added to.
     */
    attachments: Array<Attachment> | null;
    /** Messages for constructing a chat completion request **/
    messages?: ChatCompletionMessage[];
    /** Settings for constructing a chat completion request **/
    model?: ModelInfo;
    /** The thread of this message is belong to. **/
    thread?: Thread;
    /**
     * ChatCompletion tools
     */
    tools?: MessageTool[];
    /** Engine name to process */
    engine?: string;
    /** Message type */
    type?: string;
};
/**
 * ChatCompletion Tool parameters
 */
export type MessageTool = {
    type: string;
    function: MessageFunction;
};
/**
 * ChatCompletion Tool's function parameters
 */
export type MessageFunction = {
    name: string;
    description?: string;
    parameters?: Record<string, unknown>;
    strict?: boolean;
};
/**
 * The status of the message.
 * @data_transfer_object
 */
export declare enum MessageStatus {
    /** Message is fully loaded. **/
    Ready = "ready",
    /** Message is not fully loaded. **/
    Pending = "pending",
    /** Message loaded with error. **/
    Error = "error",
    /** Message is cancelled streaming */
    Stopped = "stopped"
}
export declare enum ErrorCode {
    InvalidApiKey = "invalid_api_key",
    AuthenticationError = "authentication_error",
    InsufficientQuota = "insufficient_quota",
    InvalidRequestError = "invalid_request_error",
    Unknown = "unknown"
}
/**
 * The content type of the message.
 */
export declare enum ContentType {
    Text = "text",
    Image = "image_url"
}
/**
 * The `ContentValue` type defines the shape of a content value object
 * @data_transfer_object
 */
export type ContentValue = {
    value: string;
    annotations: string[];
};
/**
 * The `ImageContentValue` type defines the shape of a content value object of image type
 * @data_transfer_object
 */
export type ImageContentValue = {
    detail?: string;
    url?: string;
};
/**
 * The `ThreadContent` type defines the shape of a message's content object
 * @data_transfer_object
 */
export type ThreadContent = {
    type: ContentType;
    text?: ContentValue;
    image_url?: ImageContentValue;
};
export interface Attachment {
    /**
     * The ID of the file to attach to the message.
     */
    file_id?: string;
    /**
     * The tools to add this file to.
     */
    tools?: Array<CodeInterpreterTool | Attachment.AssistantToolsFileSearchTypeOnly>;
}
export declare namespace Attachment {
    interface AssistantToolsFileSearchTypeOnly {
        /**
         * The type of tool being defined: `file_search`
         */
        type: 'file_search';
    }
}
/**
 * On an incomplete message, details about why the message is incomplete.
 */
export interface IncompleteDetails {
    /**
     * The reason the message is incomplete.
     */
    reason: 'content_filter' | 'max_tokens' | 'run_cancelled' | 'run_expired' | 'run_failed';
}
