import { Model, SettingComponentProps } from '../types';
export declare enum ExtensionTypeEnum {
    Assistant = "assistant",
    Conversational = "conversational",
    Inference = "inference",
    Model = "model",
    SystemMonitoring = "systemMonitoring",
    MCP = "mcp",
    HuggingFace = "huggingFace",
    Engine = "engine",
    Hardware = "hardware",
    RAG = "rag",
    VectorDB = "vectorDB"
}
export interface ExtensionType {
    type(): ExtensionTypeEnum | undefined;
}
export interface Compatibility {
    platform: string[];
    version: string;
}
/**
 * Represents a base extension.
 * This class should be extended by any class that represents an extension.
 */
export declare abstract class BaseExtension implements ExtensionType {
    protected settingFolderName: string;
    protected settingFileName: string;
    /** @type {string} Name of the extension. */
    name: string;
    /** @type {string} Product Name of the extension. */
    productName?: string;
    /** @type {string} The URL of the extension to load. */
    url: string;
    /** @type {boolean} Whether the extension is activated or not. */
    active: boolean | undefined;
    /** @type {string} Extension's description. */
    description: string | undefined;
    /** @type {string} Extension's version. */
    version: string | undefined;
    constructor(url: string, name: string, productName?: string, active?: boolean, description?: string, version?: string);
    /**
     * Returns the type of the extension.
     * @returns {ExtensionType} The type of the extension
     * Undefined means its not extending any known extension by the application.
     */
    type(): ExtensionTypeEnum | undefined;
    /**
     * Called when the extension is loaded.
     * Any initialization logic for the extension should be put here.
     */
    abstract onLoad(): void;
    /**
     * Called when the extension is unloaded.
     * Any cleanup logic for the extension should be put here.
     */
    abstract onUnload(): void;
    /**
     * The compatibility of the extension.
     * This is used to check if the extension is compatible with the current environment.
     * @property {Array} platform
     */
    compatibility(): Compatibility | undefined;
    /**
     * Registers models - it persists in-memory shared ModelManager instance's data map.
     * @param models
     */
    registerModels(models: Model[]): Promise<void>;
    /**
     * Register settings for the extension.
     * @param settings
     * @returns
     */
    registerSettings(settings: SettingComponentProps[]): Promise<void>;
    /**
     * Get the setting value for the key.
     * @param key
     * @param defaultValue
     * @returns
     */
    getSetting<T>(key: string, defaultValue: T): Promise<T>;
    onSettingUpdate<T>(key: string, value: T): void;
    /**
     * Install the prerequisites for the extension.
     *
     * @returns {Promise<void>}
     */
    install(): Promise<void>;
    /**
     * Get the settings for the extension.
     * @returns
     */
    getSettings(): Promise<SettingComponentProps[]>;
    /**
     * Update the settings for the extension.
     * @param componentProps
     * @returns
     */
    updateSettings(componentProps: Partial<SettingComponentProps>[]): Promise<void>;
}
