/**
 * Validation rules for model parameters
 */
export const validationRules = {
    temperature: (value) => typeof value === 'number' && value >= 0 && value <= 2,
    token_limit: (value) => Number.isInteger(value) && value >= 0,
    top_k: (value) => typeof value === 'number' && value >= 0,
    top_p: (value) => typeof value === 'number' && value >= 0 && value <= 1,
    stream: (value) => typeof value === 'boolean',
    max_tokens: (value) => Number.isInteger(value) && value >= 0,
    stop: (value) => Array.isArray(value) && value.every((v) => typeof v === 'string'),
    frequency_penalty: (value) => typeof value === 'number' && value >= -2 && value <= 2,
    presence_penalty: (value) => typeof value === 'number' && value >= -2 && value <= 2,
    repeat_last_n: (value) => typeof value === 'number',
    repeat_penalty: (value) => typeof value === 'number',
    min_p: (value) => typeof value === 'number',
    ctx_len: (value) => Number.isInteger(value) && value >= 0,
    ngl: (value) => Number.isInteger(value) && value >= 0,
    embedding: (value) => typeof value === 'boolean',
    n_parallel: (value) => Number.isInteger(value) && value >= 0,
    cpu_threads: (value) => Number.isInteger(value) && value >= 0,
    prompt_template: (value) => typeof value === 'string',
    llama_model_path: (value) => typeof value === 'string',
    mmproj: (value) => typeof value === 'string',
    vision_model: (value) => typeof value === 'boolean',
    text_model: (value) => typeof value === 'boolean',
};
/**
 * There are some parameters that need to be normalized before being sent to the server
 * E.g. ctx_len should be an integer, but it can be a float from the input field
 * @param key
 * @param value
 * @returns
 */
export const normalizeValue = (key, value) => {
    if (key === 'token_limit' ||
        key === 'max_tokens' ||
        key === 'ctx_len' ||
        key === 'ngl' ||
        key === 'n_parallel' ||
        key === 'cpu_threads') {
        // Convert to integer
        return Math.floor(Number(value));
    }
    if (key === 'temperature' ||
        key === 'top_k' ||
        key === 'top_p' ||
        key === 'min_p' ||
        key === 'repeat_penalty' ||
        key === 'frequency_penalty' ||
        key === 'presence_penalty' ||
        key === 'repeat_last_n') {
        // Convert to float
        const newValue = parseFloat(value);
        if (newValue !== null && !isNaN(newValue)) {
            return newValue;
        }
    }
    return value;
};
/**
 * Extract inference parameters from flat model parameters
 * @param modelParams
 * @returns
 */
export const extractInferenceParams = (modelParams, originParams) => {
    if (!modelParams)
        return {};
    const defaultModelParams = {
        temperature: undefined,
        token_limit: undefined,
        top_k: undefined,
        top_p: undefined,
        stream: undefined,
        max_tokens: undefined,
        stop: undefined,
        frequency_penalty: undefined,
        presence_penalty: undefined,
        engine: undefined,
    };
    const runtimeParams = {};
    for (const [key, value] of Object.entries(modelParams)) {
        if (key in defaultModelParams) {
            const validate = validationRules[key];
            if (validate && !validate(normalizeValue(key, value))) {
                // Invalid value - fall back to origin value
                if (originParams && key in originParams) {
                    Object.assign(runtimeParams, Object.assign(Object.assign({}, runtimeParams), { [key]: originParams[key] }));
                }
            }
            else {
                Object.assign(runtimeParams, Object.assign(Object.assign({}, runtimeParams), { [key]: normalizeValue(key, value) }));
            }
        }
    }
    return runtimeParams;
};
/**
 * Extract model load parameters from flat model parameters
 * @param modelParams
 * @returns
 */
export const extractModelLoadParams = (modelParams, originParams) => {
    if (!modelParams)
        return {};
    const defaultSettingParams = {
        ctx_len: undefined,
        ngl: undefined,
        embedding: undefined,
        n_parallel: undefined,
        cpu_threads: undefined,
        pre_prompt: undefined,
        system_prompt: undefined,
        ai_prompt: undefined,
        user_prompt: undefined,
        prompt_template: undefined,
        model_path: undefined,
        llama_model_path: undefined,
        mmproj: undefined,
        cont_batching: undefined,
        vision_model: undefined,
        text_model: undefined,
        engine: undefined,
        top_p: undefined,
        top_k: undefined,
        min_p: undefined,
        temperature: undefined,
        repeat_penalty: undefined,
        repeat_last_n: undefined,
        presence_penalty: undefined,
        frequency_penalty: undefined,
    };
    const settingParams = {};
    for (const [key, value] of Object.entries(modelParams)) {
        if (key in defaultSettingParams) {
            const validate = validationRules[key];
            if (validate && !validate(normalizeValue(key, value))) {
                // Invalid value - fall back to origin value
                if (originParams && key in originParams) {
                    Object.assign(modelParams, Object.assign(Object.assign({}, modelParams), { [key]: originParams[key] }));
                }
            }
            else {
                Object.assign(settingParams, Object.assign(Object.assign({}, settingParams), { [key]: normalizeValue(key, value) }));
            }
        }
    }
    return settingParams;
};
//# sourceMappingURL=utils.js.map