import { __awaiter } from "tslib";
import { ModelManager } from './models';
export var ExtensionTypeEnum;
(function (ExtensionTypeEnum) {
    ExtensionTypeEnum["Assistant"] = "assistant";
    ExtensionTypeEnum["Conversational"] = "conversational";
    ExtensionTypeEnum["Inference"] = "inference";
    ExtensionTypeEnum["Model"] = "model";
    ExtensionTypeEnum["SystemMonitoring"] = "systemMonitoring";
    ExtensionTypeEnum["MCP"] = "mcp";
    ExtensionTypeEnum["HuggingFace"] = "huggingFace";
    ExtensionTypeEnum["Engine"] = "engine";
    ExtensionTypeEnum["Hardware"] = "hardware";
    ExtensionTypeEnum["RAG"] = "rag";
    ExtensionTypeEnum["VectorDB"] = "vectorDB";
})(ExtensionTypeEnum || (ExtensionTypeEnum = {}));
/**
 * Represents a base extension.
 * This class should be extended by any class that represents an extension.
 */
export class BaseExtension {
    constructor(url, name, productName, active, description, version) {
        this.settingFolderName = 'settings';
        this.settingFileName = 'settings.json';
        this.name = name;
        this.productName = productName;
        this.url = url;
        this.active = active;
        this.description = description;
        this.version = version;
    }
    /**
     * Returns the type of the extension.
     * @returns {ExtensionType} The type of the extension
     * Undefined means its not extending any known extension by the application.
     */
    type() {
        return undefined;
    }
    /**
     * The compatibility of the extension.
     * This is used to check if the extension is compatible with the current environment.
     * @property {Array} platform
     */
    compatibility() {
        return undefined;
    }
    /**
     * Registers models - it persists in-memory shared ModelManager instance's data map.
     * @param models
     */
    registerModels(models) {
        return __awaiter(this, void 0, void 0, function* () {
            for (const model of models) {
                ModelManager.instance().register(model);
            }
        });
    }
    /**
     * Register settings for the extension.
     * @param settings
     * @returns
     */
    registerSettings(settings) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.name) {
                console.error('Extension name is not defined');
                return;
            }
            settings.forEach((setting) => {
                setting.extensionName = this.name;
            });
            try {
                const oldSettingsJson = localStorage.getItem(this.name);
                // Persists new settings
                if (oldSettingsJson) {
                    const oldSettings = JSON.parse(oldSettingsJson);
                    settings.forEach((setting) => {
                        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
                        // Keep setting value
                        if (setting.controllerProps && Array.isArray(oldSettings))
                            setting.controllerProps.value =
                                (_c = (_b = (_a = oldSettings.find((e) => e.key === setting.key)) === null || _a === void 0 ? void 0 : _a.controllerProps) === null || _b === void 0 ? void 0 : _b.value) !== null && _c !== void 0 ? _c : setting.controllerProps.value;
                        if ('options' in setting.controllerProps) {
                            setting.controllerProps.options = ((_d = setting.controllerProps.options) === null || _d === void 0 ? void 0 : _d.length)
                                ? setting.controllerProps.options
                                : (_f = (_e = oldSettings.find((e) => e.key === setting.key)) === null || _e === void 0 ? void 0 : _e.controllerProps) === null || _f === void 0 ? void 0 : _f.options;
                            if (!((_g = setting.controllerProps.options) === null || _g === void 0 ? void 0 : _g.some(e => e.value === setting.controllerProps.value))) {
                                setting.controllerProps.value = (_k = (_j = (_h = setting.controllerProps.options) === null || _h === void 0 ? void 0 : _h[0]) === null || _j === void 0 ? void 0 : _j.value) !== null && _k !== void 0 ? _k : setting.controllerProps.value;
                            }
                        }
                        if ('recommended' in setting.controllerProps) {
                            const oldRecommended = (_m = (_l = oldSettings.find((e) => e.key === setting.key)) === null || _l === void 0 ? void 0 : _l.controllerProps) === null || _m === void 0 ? void 0 : _m.recommended;
                            if (oldRecommended !== undefined && oldRecommended !== '') {
                                setting.controllerProps.recommended = oldRecommended;
                            }
                        }
                    });
                }
                localStorage.setItem(this.name, JSON.stringify(settings));
            }
            catch (err) {
                console.error(err);
            }
        });
    }
    /**
     * Get the setting value for the key.
     * @param key
     * @param defaultValue
     * @returns
     */
    getSetting(key, defaultValue) {
        return __awaiter(this, void 0, void 0, function* () {
            var _a;
            const keySetting = (yield this.getSettings()).find((setting) => setting.key === key);
            const value = keySetting === null || keySetting === void 0 ? void 0 : keySetting.controllerProps.value;
            return (_a = value) !== null && _a !== void 0 ? _a : defaultValue;
        });
    }
    onSettingUpdate(key, value) {
        return;
    }
    /**
     * Install the prerequisites for the extension.
     *
     * @returns {Promise<void>}
     */
    install() {
        return __awaiter(this, void 0, void 0, function* () {
            return;
        });
    }
    /**
     * Get the settings for the extension.
     * @returns
     */
    getSettings() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.name)
                return [];
            try {
                const settingsString = localStorage.getItem(this.name);
                if (!settingsString)
                    return [];
                const settings = JSON.parse(settingsString);
                return settings;
            }
            catch (err) {
                console.warn(err);
                return [];
            }
        });
    }
    /**
     * Update the settings for the extension.
     * @param componentProps
     * @returns
     */
    updateSettings(componentProps) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.name)
                return;
            const settings = yield this.getSettings();
            let updatedSettings = settings.map((setting) => {
                const updatedSetting = componentProps.find((componentProp) => componentProp.key === setting.key);
                if (updatedSetting && updatedSetting.controllerProps) {
                    setting.controllerProps.value = updatedSetting.controllerProps.value;
                }
                return setting;
            });
            if (!updatedSettings.length)
                updatedSettings = componentProps;
            localStorage.setItem(this.name, JSON.stringify(updatedSettings));
            updatedSettings.forEach((setting) => {
                this.onSettingUpdate(setting.key, setting.controllerProps.value);
            });
        });
    }
}
//# sourceMappingURL=extension.js.map